/* iksemel (XML parser for Jabber)
** Copyright (C) 2000-2004 Gurer Ozen <madcat@e-kolay.net>
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
*/

/* This file has been modified by
** Siemens Home and Office Communication Devices GmbH & Co. KG
** (SHC) 
** http://gigaset.siemens.com
** 2006-02-21: some casts
** 2007-11-19: some code optimisations
** 2008-09-24; extended ASCII coding detection
*/

#ifndef SHC_VERSION
  #include "common.h"
  #include "iksemel.h"
#endif

enum cons_e {
	C_CDATA = 0,
	C_TAG_START,
	C_TAG,
	C_TAG_END,
	C_ATTRIBUTE,
	C_ATTRIBUTE_1,
	C_ATTRIBUTE_2,
	C_VALUE,
	C_VALUE_APOS,
	C_VALUE_QUOT,
	C_WHITESPACE,
	C_ENTITY,
	C_COMMENT,
	C_COMMENT_1,
	C_COMMENT_2,
	C_COMMENT_3,
	C_MARKUP,
	C_MARKUP_1,
	C_SECT,
	C_SECT_CDATA,
	C_SECT_CDATA_1,
	C_SECT_CDATA_2,
	C_SECT_CDATA_3,
	C_SECT_CDATA_4,
	C_SECT_CDATA_C,
	C_SECT_CDATA_E,
	C_SECT_CDATA_E2,
	C_PI
};

#ifdef SHC_VERSION
typedef enum          // "Enumeration used for ASCII code recognition"
{
  IKS_ASCII_SPACE                 ,
  IKS_ASCII_SPACE_CODE            ,
  IKS_ASCII_QUOT                  ,
  IKS_ASCII_AND                   ,
  IKS_ASCII_EXCLAMATION           ,
  IKS_ASCII_QUOTATION             ,
  IKS_ASCII_QUOTATION_8220        ,
  IKS_ASCII_APOSTROPHE            ,
  IKS_ASCII_APOSTROPHE_CODE       ,
  IKS_ASCII_QUOTATION_8221        ,  
  IKS_ASCII_HELLIP_NUM            ,
  IKS_ASCII_HELLIP                ,  
  IKS_ASCII_HASH                  ,
  IKS_ASCII_DOLLAR                ,
  IKS_ASCII_PERCENT               ,
  IKS_ASCII_AND_NUM               ,
  IKS_ASCII_ACCENT                ,
  IKS_ASCII_LEFT_PARENTHESIS      ,
  IKS_ASCII_RIGHT_PARENTHESIS     ,
  IKS_ASCII_ASTERISK              ,
  IKS_ASCII_PLUS                  ,
  IKS_ASCII_COMMA                 ,
  IKS_ASCII_HYPHEN_MINUS          ,
  IKS_ASCII_FULL_STOP             ,
  IKS_ASCII_SOLIDUS               ,
  IKS_ASCII_DIGIT_ZERO            ,
  IKS_ASCII_DIGIT_ONE             ,
  IKS_ASCII_DIGIT_TWO             ,
  IKS_ASCII_DIGIT_THREE           ,
  IKS_ASCII_DIGIT_FOUR            ,
  IKS_ASCII_DIGIT_FIVE            ,
  IKS_ASCII_DIGIT_SIX             ,
  IKS_ASCII_DIGIT_SEVEN           ,
  IKS_ASCII_DIGIT_EIGHT           ,
  IKS_ASCII_DIGIT_NINE            ,
  IKS_ASCII_COLON                 ,
  IKS_ASCII_SEMICOLON             ,
  IKS_ASCII_LESS_THAN             , IKS_ASCII_LESS_THAN_CODE        ,
  IKS_ASCII_EQUAL                 ,
  IKS_ASCII_GREATER_THAN          , IKS_ASCII_GREATER_THAN_CODE     ,
  IKS_ASCII_QUESTION              ,
  IKS_ASCII_AT                    ,
  IKS_ASCII_A_CAP                 ,
  IKS_ASCII_B_CAP                 ,
  IKS_ASCII_C_CAP                 ,
  IKS_ASCII_D_CAP                 ,
  IKS_ASCII_E_CAP                 ,
  IKS_ASCII_F_CAP                 ,
  IKS_ASCII_G_CAP                 ,
  IKS_ASCII_H_CAP                 ,
  IKS_ASCII_I_CAP                 ,
  IKS_ASCII_J_CAP                 ,
  IKS_ASCII_K_CAP                 ,
  IKS_ASCII_l_CAP                 ,
  IKS_ASCII_M_CAP                 ,
  IKS_ASCII_N_CAP                 ,
  IKS_ASCII_O_CAP                 ,
  IKS_ASCII_P_CAP                 ,
  IKS_ASCII_Q_CAP                 ,
  IKS_ASCII_R_CAP                 ,
  IKS_ASCII_S_CAP                 ,
  IKS_ASCII_T_CAP                 ,
  IKS_ASCII_U_CAP                 ,
  IKS_ASCII_V_CAP                 ,
  IKS_ASCII_W_CAP                 ,
  IKS_ASCII_X_CAP                 ,
  IKS_ASCII_Y_CAP                 ,
  IKS_ASCII_Z_CAP                 ,
  IKS_ASCII_LEFT_SQUARE_BRACKET   ,
  IKS_ASCII_SOLIDUS_REVERSE       ,
  IKS_ASCII_RIGHT_SQUARE_BRACKET  ,
  IKS_ASCII_CIRCUMFLEX_ACCENT     ,
  IKS_ASCII_LOW_LINE              ,
  IKS_ASCII_GRAVE_ACCENT          ,
  IKS_ASCII_A                     ,
  IKS_ASCII_B                     ,
  IKS_ASCII_C                     ,
  IKS_ASCII_D                     ,
  IKS_ASCII_E                     ,
  IKS_ASCII_F                     ,
  IKS_ASCII_G                     ,
  IKS_ASCII_H                     ,
  IKS_ASCII_I                     ,
  IKS_ASCII_J                     ,
  IKS_ASCII_K                     ,
  IKS_ASCII_L                     ,
  IKS_ASCII_M                     ,
  IKS_ASCII_N                     ,
  IKS_ASCII_O                     ,
  IKS_ASCII_P                     ,
  IKS_ASCII_Q                     ,
  IKS_ASCII_R                     ,
  IKS_ASCII_S                     ,
  IKS_ASCII_T                     ,
  IKS_ASCII_U                     ,
  IKS_ASCII_V                     ,
  IKS_ASCII_W                     ,
  IKS_ASCII_X                     ,
  IKS_ASCII_Y                     ,
  IKS_ASCII_Z                     ,
  IKS_ASCII_LEFT_CURLY_BRACKET    ,
  IKS_ASCII_VERTICAL_LINE         ,
  IKS_ASCII_RIGHT_CURLY_BRACKET   ,
  IKS_ASCII_TILDE                 ,
  IKS_ASCII_DELETE                ,
  IKS_ASCII_EURO                  , IKS_ASCII_EURO_CODE             ,
  IKS_ASCII_EURO_CAP_CODE         , 
  IKS_ASCII_EURO_NUM_CODE         ,
  IKS_ASCII_ACCENT1               ,
  IKS_ASCII_ACCENT2               ,
  IKS_ASCII_NBSP_CHAR             ,
  IKS_ASCII_INV_EXCLAMATION_MARK  ,
  IKS_ASCII_CENT                  ,
  IKS_ASCII_POUND                 ,
  IKS_ASCII_CURRENCY              ,
  IKS_ASCII_YEN                   ,
  IKS_ASCII_BROKEN_BAR            ,
  IKS_ASCII_SECTION               ,
  IKS_ASCII_DIAERESIS             ,
  IKS_ASCII_COPYRIGHT             , IKS_ASCII_COPYRIGHT_CODE        ,
  IKS_ASCII_FEMININE              ,
  IKS_ASCII_LEFT_DOUBLE_ANGLE     ,
  IKS_ASCII_NOT_SIGN              ,
  IKS_ASCII_SOFT_HYPHEN           ,
  IKS_ASCII_REGISTERED            , IKS_ASCII_REGISTERED_CODE       ,
  IKS_ASCII_MACRON                ,
  IKS_ASCII_DEGREE                ,
  IKS_ASCII_PLUS_MINUS            ,
  IKS_ASCII_SUPERSCRIPT_TWO       ,
  IKS_ASCII_SUPERSCRIPT_THREE     ,
  IKS_ASCII_ACUTE_ACCENT          ,
  IKS_ASCII_MICRO                 ,
  IKS_ASCII_PILCROW               ,
  IKS_ASCII_MIDDLE_DOT            ,
  IKS_ASCII_CEDILLA               ,
  IKS_ASCII_SUPERSCRIPT_1         ,
  IKS_ASCII_MASCULINE             ,
  IKS_ASCII_RICHT_DOUBLE_ANGLE    ,         
  IKS_ASCII_ONE_QUARTER           ,
  IKS_ASCII_ONE_HALF              ,
  IKS_ASCII_THREE_QUARTERS        ,
  IKS_ASCII_INV_QUESTION          ,
  IKS_ASCII_A_GRAVE_CAP           , IKS_ASCII_A_GRAVE_CAP_CODE      ,
  IKS_ASCII_A_ACUTE_CAP           , IKS_ASCII_A_ACUTE_CAP_CODE      ,
  IKS_ASCII_A_CIRCUMFLEX_CAP      , IKS_ASCII_A_CIRCUMFLEX_CAP_CODE ,
  IKS_ASCII_A_TILDE_CAP           , IKS_ASCII_A_TILDE_CAP_CODE      ,
  IKS_ASCII_A_DIAERESIS_CAP       , IKS_ASCII_A_DIAERESIS_CAP_CODE  ,
  IKS_ASCII_A_RING_CAP            , IKS_ASCII_A_RING_CAP_CODE       ,
  IKS_ASCII_AE_CAP                , IKS_ASCII_AE_CAP_CODE           ,
  IKS_ASCII_C_CEDILLA_CAP         , IKS_ASCII_C_CEDILLA_CAP_CODE    ,
  IKS_ASCII_E_GRAVE_CAP           , IKS_ASCII_E_GRAVE_CAP_CODE      ,
  IKS_ASCII_E_ACUTE_CAP           , IKS_ASCII_E_ACUTE_CAP_CODE      ,
  IKS_ASCII_E_CIRCUMFLEX_CAP      , IKS_ASCII_E_CIRCUMFLEX_CAP_CODE ,
  IKS_ASCII_E_DIAERESIS_CAP       , IKS_ASCII_E_DIAERESIS_CAP_CODE  ,
  IKS_ASCII_I_GRAVE_CAP           , IKS_ASCII_I_GRAVE_CAP_CODE      ,
  IKS_ASCII_I_ACUTE_CAP           , IKS_ASCII_I_ACUTE_CAP_CODE      ,
  IKS_ASCII_I_CIRCUMFLEX_CAP      , IKS_ASCII_I_CIRCUMFLEX_CAP_CODE ,
  IKS_ASCII_I_DIAERESIS_CAP       , IKS_ASCII_I_DIAERESIS_CAP_CODE  ,
  IKS_ASCII_ETH_CAP               , IKS_ASCII_ETH_CAP_CODE          ,
  IKS_ASCII_N_TILDE_CAP           , IKS_ASCII_N_TILDE_CAP_CODE      ,
  IKS_ASCII_O_GRAVE_CAP           , IKS_ASCII_O_GRAVE_CAP_CODE      ,
  IKS_ASCII_O_ACUTE_CAP           , IKS_ASCII_O_ACUTE_CAP_CODE      ,
  IKS_ASCII_O_CIRCUMFLEX_CAP      , IKS_ASCII_O_CIRCUMFLEX_CAP_CODE ,
  IKS_ASCII_O_TILDE_CAP           , IKS_ASCII_O_TILDE_CAP_CODE      ,
  IKS_ASCII_O_DIAERESIS_CAP       , IKS_ASCII_O_DIAERESIS_CAP_CODE  ,
  IKS_ASCII_O_MULTIPLICATION      ,
  IKS_ASCII_O_STROKE_CAP          , IKS_ASCII_O_STROKE_CAP_CODE     ,
  IKS_ASCII_U_GRAVE_CAP           , IKS_ASCII_U_GRAVE_CAP_CODE      ,
  IKS_ASCII_U_ACUTE_CAP           , IKS_ASCII_U_ACUTE_CAP_CODE      ,
  IKS_ASCII_U_CIRCUMFLEX_CAP      , IKS_ASCII_U_CIRCUMFLEX_CAP_CODE ,
  IKS_ASCII_U_DIAERESIS_CAP       , IKS_ASCII_U_DIAERESIS_CAP_CODE  ,
  IKS_ASCII_Y_ACUTE_CAP           , IKS_ASCII_Y_ACUTE_CAP_CODE      ,
  IKS_ASCII_THORN_CAP             , IKS_ASCII_THORN_CAP_CODE        ,
  IKS_ASCII_SHARP_S               , IKS_ASCII_SHARP_S_CODE          ,
  IKS_ASCII_A_GRAVE               , IKS_ASCII_A_GRAVE_CODE          ,
  IKS_ASCII_A_ACUTE               , IKS_ASCII_A_ACUTE_CODE          ,
  IKS_ASCII_A_CIRCUMFLEX          , IKS_ASCII_A_CIRCUMFLEX_CODE     ,
  IKS_ASCII_A_TILDE               , IKS_ASCII_A_TILDE_CODE          ,
  IKS_ASCII_A_DIAERESIS           , IKS_ASCII_A_DIAERESIS_CODE      ,
  IKS_ASCII_A_RING                , IKS_ASCII_A_RING_CODE           ,
  IKS_ASCII_AE                    , IKS_ASCII_AE_CODE               ,
  IKS_ASCII_C_CEDILLA             , IKS_ASCII_C_CEDILLA_CODE        ,
  IKS_ASCII_E_GRAVE               , IKS_ASCII_E_GRAVE_CODE          ,
  IKS_ASCII_E_ACUTE               , IKS_ASCII_E_ACUTE_CODE          ,
  IKS_ASCII_E_CIRCUMFLEX          , IKS_ASCII_E_CIRCUMFLEX_CODE     ,
  IKS_ASCII_E_DIAERESIS           , IKS_ASCII_E_DIAERESIS_CODE      ,
  IKS_ASCII_I_GRAVE               , IKS_ASCII_I_GRAVE_CODE          ,
  IKS_ASCII_I_ACUTE               , IKS_ASCII_I_ACUTE_CODE          ,
  IKS_ASCII_I_CIRCUMFLEX          , IKS_ASCII_I_CIRCUMFLEX_CODE     ,
  IKS_ASCII_I_DIAERESIS           , IKS_ASCII_I_DIAERESIS_CODE      ,
  IKS_ASCII_ETH                   , IKS_ASCII_ETH_CODE              ,
  IKS_ASCII_N_TILDE               , IKS_ASCII_N_TILDE_CODE          ,
  IKS_ASCII_O_GRAVE               , IKS_ASCII_O_GRAVE_CODE          ,
  IKS_ASCII_O_ACUTE               , IKS_ASCII_O_ACUTE_CODE          ,
  IKS_ASCII_O_CIRCUMFLEX          , IKS_ASCII_O_CIRCUMFLEX_CODE     ,
  IKS_ASCII_O_TILDE               , IKS_ASCII_O_TILDE_CODE          ,
  IKS_ASCII_O_DIAERESIS           , IKS_ASCII_O_DIAERESIS_CODE      ,
  IKS_ASCII_DIVISION              ,
  IKS_ASCII_O_STROKE              , IKS_ASCII_O_STROKE_CODE         ,
  IKS_ASCII_U_GRAVE               , IKS_ASCII_U_GRAVE_CODE          ,
  IKS_ASCII_U_ACUTE               , IKS_ASCII_U_ACUTE_CODE          ,
  IKS_ASCII_U_CIRCUMFLEX          , IKS_ASCII_U_CIRCUMFLEX_CODE     ,
  IKS_ASCII_U_DIAERESIS           , IKS_ASCII_U_DIAERESIS_CODE      ,
  IKS_ASCII_Y_ACUTE               , IKS_ASCII_Y_ACUTE_CODE          ,
  IKS_ASCII_THORN                 , IKS_ASCII_THORN_CODE            ,
  IKS_ASCII_Y_DIAERESIS           , IKS_ASCII_Y_DIAERESIS_CODE      ,
  IKS_ASCII_UNKNOWN   
} IKS_ASCII_NO;

typedef struct      // "Structure used for ASCII code table"
{
  IKS_ASCII_NO eNo;
  char *pcAsciiCode;
  char *pcNumCode;
  char cChar;
} IKS_ASCII_CODE_DEFINITION;

static IKS_ASCII_CODE_DEFINITION astAsciiCodeDefinitions[] =
{ //Character code,                Character,            URL encoded
  {IKS_ASCII_SPACE                 ,"&nbsp;"   ,"nbsp;"   , ' '  },
  {IKS_ASCII_SPACE_CODE            ,"&#32;"    ,"#32;"    , ' '  },
  {IKS_ASCII_QUOT                  ,"&quot;"   ,"quot;"   , (char) 0x22 },    // "
  {IKS_ASCII_AND                   ,"&amp;"    ,"amp;"    , '&'  }, 
  {IKS_ASCII_EXCLAMATION           ,"&#33;"    ,"#33;"    , '!'  },
  {IKS_ASCII_QUOTATION             ,"&#34;"    ,"#34;"    , (char) 0x22  },
  {IKS_ASCII_QUOTATION_8220        ,"&#8220;"  ,"#8220;"  , (char) 0x22  },
  {IKS_ASCII_APOSTROPHE            ,"&#8217;"  ,"#8217;"  , (char) 0x27  },
  {IKS_ASCII_APOSTROPHE_CODE       ,"&apos;"   ,"apos;"   , (char) 0x27  },
  {IKS_ASCII_QUOTATION_8221        ,"&#8221;"  ,"#8221;"  , (char) 0x22  },
  {IKS_ASCII_HELLIP_NUM            ,"&#8230;"  ,"#8230;"  , (char) 0x22  },
  {IKS_ASCII_HELLIP                ,"&hellip;" ,"hellip;" , (char) 0x22  },
  {IKS_ASCII_HASH                  ,"&#35;"    ,"#35;"    , '#'  }, 
  {IKS_ASCII_DOLLAR                ,"&#36;"    ,"#36;"    , '$'  }, 
  {IKS_ASCII_PERCENT               ,"&#37;"    ,"#37;"    , '%'  }, 
  {IKS_ASCII_AND_NUM               ,"&#38;"    ,"#38;"    , '&'  },  
  {IKS_ASCII_ACCENT                ,"&#39;"    ,"#39;"    , (char) 0x27  },  
  {IKS_ASCII_LEFT_PARENTHESIS      ,"&#40;"    ,"#40;"    , '('  },   
  {IKS_ASCII_RIGHT_PARENTHESIS     ,"&#41;"    ,"#41;"    , ')'  },    
  {IKS_ASCII_ASTERISK              ,"&#42;"    ,"#42;"    , '*'  }, 
  {IKS_ASCII_PLUS                  ,"&#43;"    ,"#43;"    , '+'  },
  {IKS_ASCII_COMMA                 ,"&#44;"    ,"#44;"    , ','  },
  {IKS_ASCII_HYPHEN_MINUS          ,"&#45;"    ,"#45;"    , '-'  },
  {IKS_ASCII_FULL_STOP             ,"&#46;"    ,"#46;"    , '.'  },
  {IKS_ASCII_SOLIDUS               ,"&#47;"    ,"#47;"    , '/'  },
  {IKS_ASCII_DIGIT_ZERO            ,"&#48;"    ,"#48;"    , '0'  },
  {IKS_ASCII_DIGIT_ONE             ,"&#49;"    ,"#49;"    , '1'  },
  {IKS_ASCII_DIGIT_TWO             ,"&#50;"    ,"#50;"    , '2'  },
  {IKS_ASCII_DIGIT_THREE           ,"&#51;"    ,"#51;"    , '3'  },
  {IKS_ASCII_DIGIT_FOUR            ,"&#52;"    ,"#52;"    , '4'  },
  {IKS_ASCII_DIGIT_FIVE            ,"&#53;"    ,"#53;"    , '5'  },
  {IKS_ASCII_DIGIT_SIX             ,"&#54;"    ,"#54;"    , '6'  },
  {IKS_ASCII_DIGIT_SEVEN           ,"&#55;"    ,"#55;"    , '7'  },
  {IKS_ASCII_DIGIT_EIGHT           ,"&#56;"    ,"#56;"    , '8'  },
  {IKS_ASCII_DIGIT_NINE            ,"&#57;"    ,"#57;"    , '9'  },
  {IKS_ASCII_COLON                 ,"&#58;"    ,"#58;"    , ':'  },
  {IKS_ASCII_SEMICOLON             ,"&#59;"    ,"#59;"    , ';'  },
  {IKS_ASCII_LESS_THAN             ,"&#60;"    ,"#60;"    , '<'  },
  {IKS_ASCII_LESS_THAN_CODE        ,"&lt;"     ,"lt;"     , '<'  },  
  {IKS_ASCII_EQUAL                 ,"&#61;"    ,"#61;"    , '='  },
  {IKS_ASCII_GREATER_THAN          ,"&#62;"    ,"#62;"    , '>'  }, 
  {IKS_ASCII_GREATER_THAN_CODE     ,"&gt;"     ,"gt;"     , '>'  },  
  {IKS_ASCII_QUESTION              ,"&#63;"    ,"#63;"    , '?'  },
  {IKS_ASCII_AT                    ,"&#64;"    ,"#64;"    , '@'  },
  {IKS_ASCII_A_CAP                 ,"&#65;"    ,"#65;"    , 'A'  }, 
  {IKS_ASCII_B_CAP                 ,"&#66;"    ,"#66;"    , 'B'  }, 
  {IKS_ASCII_C_CAP                 ,"&#67;"    ,"#67;"    , 'C'  }, 
  {IKS_ASCII_D_CAP                 ,"&#68;"    ,"#68;"    , 'D'  }, 
  {IKS_ASCII_E_CAP                 ,"&#69;"    ,"#69;"    , 'E'  }, 
  {IKS_ASCII_F_CAP                 ,"&#70;"    ,"#70;"    , 'F'  }, 
  {IKS_ASCII_G_CAP                 ,"&#71;"    ,"#71;"    , 'G'  }, 
  {IKS_ASCII_H_CAP                 ,"&#72;"    ,"#72;"    , 'H'  }, 
  {IKS_ASCII_I_CAP                 ,"&#73;"    ,"#73;"    , 'I'  }, 
  {IKS_ASCII_J_CAP                 ,"&#74;"    ,"#74;"    , 'J'  }, 
  {IKS_ASCII_K_CAP                 ,"&#75;"    ,"#75;"    , 'K'  }, 
  {IKS_ASCII_l_CAP                 ,"&#76;"    ,"#76;"    , 'L'  }, 
  {IKS_ASCII_M_CAP                 ,"&#77;"    ,"#77;"    , 'M'  }, 
  {IKS_ASCII_N_CAP                 ,"&#78;"    ,"#78;"    , 'N'  }, 
  {IKS_ASCII_O_CAP                 ,"&#79;"    ,"#79;"    , 'O'  }, 
  {IKS_ASCII_P_CAP                 ,"&#80;"    ,"#80;"    , 'P'  }, 
  {IKS_ASCII_Q_CAP                 ,"&#81;"    ,"#81;"    , 'Q'  }, 
  {IKS_ASCII_R_CAP                 ,"&#82;"    ,"#82;"    , 'R'  }, 
  {IKS_ASCII_S_CAP                 ,"&#83;"    ,"#83;"    , 'S'  }, 
  {IKS_ASCII_T_CAP                 ,"&#84;"    ,"#84;"    , 'T'  }, 
  {IKS_ASCII_U_CAP                 ,"&#85;"    ,"#85;"    , 'U'  }, 
  {IKS_ASCII_V_CAP                 ,"&#86;"    ,"#86;"    , 'V'  }, 
  {IKS_ASCII_W_CAP                 ,"&#87;"    ,"#87;"    , 'W'  }, 
  {IKS_ASCII_X_CAP                 ,"&#88;"    ,"#88;"    , 'X'  }, 
  {IKS_ASCII_Y_CAP                 ,"&#89;"    ,"#89;"    , 'Y'  }, 
  {IKS_ASCII_Z_CAP                 ,"&#90;"    ,"#90;"    , 'Z'  }, 
  {IKS_ASCII_LEFT_SQUARE_BRACKET   ,"&#91;"    ,"#91;"    , '['  }, 
  {IKS_ASCII_SOLIDUS_REVERSE       ,"&#92;"    ,"#92;"    , (char) 0x5C  }, 
  {IKS_ASCII_RIGHT_SQUARE_BRACKET  ,"&#93;"    ,"#93;"    , ']'  }, 
  {IKS_ASCII_CIRCUMFLEX_ACCENT     ,"&#94;"    ,"#94;"    , '^'  }, 
  {IKS_ASCII_LOW_LINE              ,"&#95;"    ,"#95;"    , '_'  },
  {IKS_ASCII_GRAVE_ACCENT          ,"&#96;"    ,"#96;"    , '`'  },
  {IKS_ASCII_A                     ,"&#97;"    ,"#97;"    , 'a'  }, 
  {IKS_ASCII_B                     ,"&#98;"    ,"#98;"    , 'b'  }, 
  {IKS_ASCII_C                     ,"&#99;"    ,"#99;"    , 'c'  }, 
  {IKS_ASCII_D                     ,"&#100;"   ,"#100;"   , 'd'  },  
  {IKS_ASCII_E                     ,"&#101;"   ,"#101;"   , 'e'  },  
  {IKS_ASCII_F                     ,"&#102;"   ,"#102;"   , 'f'  },   
  {IKS_ASCII_G                     ,"&#103;"   ,"#103;"   , 'g'  },    
  {IKS_ASCII_H                     ,"&#104;"   ,"#104;"   , 'h'  },  
  {IKS_ASCII_I                     ,"&#105;"   ,"#105;"   , 'i'  },  
  {IKS_ASCII_J                     ,"&#106;"   ,"#106;"   , 'j'  },  
  {IKS_ASCII_K                     ,"&#107;"   ,"#107;"   , 'k'  },  
  {IKS_ASCII_L                     ,"&#108;"   ,"#108;"   , 'l'  },  
  {IKS_ASCII_M                     ,"&#109;"   ,"#109;"   , 'm'  },  
  {IKS_ASCII_N                     ,"&#110;"   ,"#110;"   , 'n'  },  
  {IKS_ASCII_O                     ,"&#111;"   ,"#111;"   , 'o'  },  
  {IKS_ASCII_P                     ,"&#112;"   ,"#112;"   , 'p'  },  
  {IKS_ASCII_Q                     ,"&#113;"   ,"#113;"   , 'q'  },  
  {IKS_ASCII_R                     ,"&#114;"   ,"#114;"   , 'r'  },  
  {IKS_ASCII_S                     ,"&#115;"   ,"#115;"   , 's'  },  
  {IKS_ASCII_T                     ,"&#116;"   ,"#116;"   , 't'  },  
  {IKS_ASCII_U                     ,"&#117;"   ,"#117;"   , 'u'  },  
  {IKS_ASCII_V                     ,"&#118;"   ,"#118;"   , 'v'  },  
  {IKS_ASCII_W                     ,"&#119;"   ,"#119;"   , 'w'  },  
  {IKS_ASCII_X                     ,"&#120;"   ,"#120;"   , 'x'  },  
  {IKS_ASCII_Y                     ,"&#121;"   ,"#121;"   , 'y'  },  
  {IKS_ASCII_Z                     ,"&#122;"   ,"#122;"   , 'z'  },  
  {IKS_ASCII_LEFT_CURLY_BRACKET    ,"&#123;"   ,"#123;"   , '{'  },   
  {IKS_ASCII_VERTICAL_LINE         ,"&#124;"   ,"#124;"   , '|'  },   
  {IKS_ASCII_RIGHT_CURLY_BRACKET   ,"&#125;"   ,"#125;"   , '}'  },   
  {IKS_ASCII_TILDE                 ,"&#126;"   ,"#126;"   , '~'  },   
  {IKS_ASCII_DELETE                ,"&#127;"   ,"#127;"   , (char) 0x7f  },   
  {IKS_ASCII_EURO                  ,"&#128;"   ,"#128;"   , (char) 0x80  },   
  {IKS_ASCII_EURO_CODE             ,"&euro;"   ,"euro;"   , (char) 0x80  },   
  {IKS_ASCII_EURO_CAP_CODE         ,"&EURO;"   ,"EURO;"   , (char) 0x80  },   
  {IKS_ASCII_EURO_NUM_CODE         ,"&#8364;"  ,"#8364;"  , (char) 0x80  },   

  {IKS_ASCII_ACCENT1               ,"&#145;"   ,"#145;"   , (char) 0x91  },   
  {IKS_ASCII_ACCENT2               ,"&#146;"   ,"#146;"   , (char) 0x92  },   

  {IKS_ASCII_NBSP_CHAR             ,"&#160;"   ,"#160;"   , (char) 0xA0  },
  {IKS_ASCII_INV_EXCLAMATION_MARK  ,"&#161;"   ,"#161;"   , (char)''  }, 
  {IKS_ASCII_CENT                  ,"&#162;"   ,"#162;"   , (char)''  },  
  {IKS_ASCII_POUND                 ,"&#163;"   ,"#163;"   , (char)''  },  
  {IKS_ASCII_CURRENCY              ,"&#164;"   ,"#164;"   , (char)''  },    
  {IKS_ASCII_YEN                   ,"&#165;"   ,"#165;"   , (char)''  },     
  {IKS_ASCII_BROKEN_BAR            ,"&#166;"   ,"#166;"   , (char)''  },  
  {IKS_ASCII_SECTION               ,"&#167;"   ,"#167;"   , (char)''  },  
  {IKS_ASCII_DIAERESIS             ,"&#168;"   ,"#168;"   , (char)''  },  
  {IKS_ASCII_COPYRIGHT             ,"&#169;"   ,"#169;"   , (char)''  },  
  {IKS_ASCII_COPYRIGHT_CODE        ,"&copy;"   ,"copy;"   , (char)''  },
  {IKS_ASCII_FEMININE              ,"&#170;"   ,"#170;"   , (char)''  },  
  {IKS_ASCII_LEFT_DOUBLE_ANGLE     ,"&#171;"   ,"#171;"   , (char)''  },  
  {IKS_ASCII_NOT_SIGN              ,"&#172;"   ,"#172;"   , (char)''  },  
  {IKS_ASCII_SOFT_HYPHEN           ,"&#173;"   ,"#173;"   , (char)''  },  
  {IKS_ASCII_REGISTERED            ,"&#174;"   ,"#174;"   , (char)''  },  
  {IKS_ASCII_REGISTERED_CODE       ,"&reg;"    ,"reg;"    , (char)''  },
  {IKS_ASCII_MACRON                ,"&#175;"   ,"#175;"   , (char)''  },  
  {IKS_ASCII_DEGREE                ,"&#176;"   ,"#176;"   , (char)''  },  
  {IKS_ASCII_PLUS_MINUS            ,"&#177;"   ,"#177;"   , (char)''  },
  {IKS_ASCII_SUPERSCRIPT_TWO       ,"&#178;"   ,"#178;"   , (char)''  },  
  {IKS_ASCII_SUPERSCRIPT_THREE     ,"&#179;"   ,"#179;"   , (char)''  },  
  {IKS_ASCII_ACUTE_ACCENT          ,"&#180;"   ,"#180;"   , (char)''  },
  {IKS_ASCII_MICRO                 ,"&#181;"   ,"#181;"   , (char)''  }, 
  {IKS_ASCII_PILCROW               ,"&#182;"   ,"#182;"   , (char)''  }, 
  {IKS_ASCII_MIDDLE_DOT            ,"&#183;"   ,"#183;"   , (char)''  }, 
  {IKS_ASCII_CEDILLA               ,"&#184;"   ,"#184;"   , (char)''  }, 
  {IKS_ASCII_SUPERSCRIPT_1         ,"&#185;"   ,"#185;"   , (char)''  }, 
  {IKS_ASCII_MASCULINE             ,"&#186;"   ,"#186;"   , (char)''  }, 
  {IKS_ASCII_RICHT_DOUBLE_ANGLE    ,"&#187;"   ,"#187;"   , (char)''  }, 
  {IKS_ASCII_ONE_QUARTER           ,"&#188;"   ,"#188;"   , (char)''  }, 
  {IKS_ASCII_ONE_HALF              ,"&#189;"   ,"#189;"   , (char)''  }, 
  {IKS_ASCII_THREE_QUARTERS        ,"&#190;"   ,"#190;"   , (char)''  }, 
  {IKS_ASCII_INV_QUESTION          ,"&#191;"   ,"#191;"   , (char)''  }, 
  {IKS_ASCII_A_GRAVE_CAP           ,"&#192;"   ,"#192;"   , (char)''  }, 
  {IKS_ASCII_A_GRAVE_CAP_CODE      ,"&Agrave;" ,"Agrave;" , (char)''  },  
  {IKS_ASCII_A_ACUTE_CAP           ,"&#193;"   ,"#193;"   , (char)''  }, 
  {IKS_ASCII_A_ACUTE_CAP_CODE      ,"&Aacute;" ,"Aacute;" , (char)''  },    
  {IKS_ASCII_A_CIRCUMFLEX_CAP      ,"&#194;"   ,"#194;"   , (char)''  },     
  {IKS_ASCII_A_CIRCUMFLEX_CAP_CODE ,"&Acirc;"  ,"Acirc;"  , (char)''  },                
  {IKS_ASCII_A_TILDE_CAP           ,"&#195;"   ,"#195;"   , (char)''  },     
  {IKS_ASCII_A_TILDE_CAP_CODE      ,"&Atilde;" ,"Atilde;" , (char)''  },     
  {IKS_ASCII_A_DIAERESIS_CAP       ,"&#196;"   ,"#196;"   , (char)''  },    
  {IKS_ASCII_A_DIAERESIS_CAP_CODE  ,"&Auml;"   ,"Auml;"   , (char)''  },    
  {IKS_ASCII_A_RING_CAP            ,"&#197;"   ,"#197;"   , (char)''  },     
  {IKS_ASCII_A_RING_CAP_CODE       ,"&Aring;"  ,"Aring;"  , (char)''  },     
  {IKS_ASCII_AE_CAP                ,"&#198;"   ,"#198;"   , (char)''  },     
  {IKS_ASCII_AE_CAP_CODE           ,"&AElig;"  ,"AElig;"  , (char)''  },     
  {IKS_ASCII_C_CEDILLA_CAP         ,"&#199;"   ,"#199;"   , (char)''  },     
  {IKS_ASCII_C_CEDILLA_CAP_CODE    ,"&Ccedil;" ,"Ccedil;" , (char)''  },     
  {IKS_ASCII_E_GRAVE_CAP           ,"&#200;"   ,"#200;"   , (char)''  },     
  {IKS_ASCII_E_GRAVE_CAP_CODE      ,"&Egrave;" ,"Egrave;" , (char)''  },     
  {IKS_ASCII_E_ACUTE_CAP           ,"&#201;"   ,"#201;"   , (char)''  },     
  {IKS_ASCII_E_ACUTE_CAP_CODE      ,"&Eacute;" ,"Eacute;" , (char)''  },      
  {IKS_ASCII_E_CIRCUMFLEX_CAP      ,"&#202;"   ,"#202;"   , (char)''  },      
  {IKS_ASCII_E_CIRCUMFLEX_CAP_CODE ,"&Ecirc;"  ,"Ecirc;"  , (char)''  },      
  {IKS_ASCII_E_DIAERESIS_CAP       ,"&#203;"   ,"#203;"   , (char)''  },     
  {IKS_ASCII_E_DIAERESIS_CAP_CODE  ,"&Euml;"   ,"Euml;"   , (char)''  },      
  {IKS_ASCII_I_GRAVE_CAP           ,"&#204;"   ,"#204;"   , (char)''  },     
  {IKS_ASCII_I_GRAVE_CAP_CODE      ,"&Igrave;" ,"Igrave;" , (char)''  },     
  {IKS_ASCII_I_ACUTE_CAP           ,"&#205;"   ,"#205;"   , (char)''  },     
  {IKS_ASCII_I_ACUTE_CAP_CODE      ,"&Iacute;" ,"Iacute;" , (char)''  },     
  {IKS_ASCII_I_CIRCUMFLEX_CAP      ,"&#206;"   ,"#206;"   , (char)''  },     
  {IKS_ASCII_I_CIRCUMFLEX_CAP_CODE ,"&Icirc;"  ,"Icirc;"  , (char)''  },     
  {IKS_ASCII_I_DIAERESIS_CAP       ,"&#207;"   ,"#207;"   , (char)''  },     
  {IKS_ASCII_I_DIAERESIS_CAP_CODE  ,"&Iuml;"   ,"Iuml;"   , (char)''  },     
  {IKS_ASCII_ETH_CAP               ,"&#208;"   ,"#208;"   , (char)''  },     
  {IKS_ASCII_ETH_CAP_CODE          ,"&ETH;"    ,"ETH;"    , (char)''  },     
  {IKS_ASCII_N_TILDE_CAP           ,"&#209;"   ,"#209;"   , (char)''  },     
  {IKS_ASCII_N_TILDE_CAP_CODE      ,"&Ntilde;" ,"Ntilde;" , (char)''  },     
  {IKS_ASCII_O_GRAVE_CAP           ,"&#210;"   ,"#210;"   , (char)''  },     
  {IKS_ASCII_O_GRAVE_CAP_CODE      ,"&Ograve;" ,"Ograve;" , (char)''  },     
  {IKS_ASCII_O_ACUTE_CAP           ,"&#211;"   ,"#211;"   , (char)''  },     
  {IKS_ASCII_O_ACUTE_CAP_CODE      ,"&Oacute;" ,"Oacute;" , (char)''  },     
  {IKS_ASCII_O_CIRCUMFLEX_CAP      ,"&#212;"   ,"#212;"   , (char)''  },     
  {IKS_ASCII_O_CIRCUMFLEX_CAP_CODE ,"&Ocirc;"  ,"Ocirc;"  , (char)''  },     
  {IKS_ASCII_O_TILDE_CAP           ,"&#213;"   ,"#213;"   , (char)''  },     
  {IKS_ASCII_O_TILDE_CAP_CODE      ,"&Otilde;" ,"Otilde;" , (char)''  }, 
  {IKS_ASCII_O_DIAERESIS_CAP       ,"&#214;"   ,"#214;"   , (char)''  },     
  {IKS_ASCII_O_DIAERESIS_CAP_CODE  ,"&Ouml;"   ,"Ouml;"   , (char)''  },                
  {IKS_ASCII_O_MULTIPLICATION      ,"&#215;"   ,"#215;"   , (char)''  },     
  {IKS_ASCII_O_STROKE_CAP          ,"&#216;"   ,"#216;"   , (char)''  },     
  {IKS_ASCII_O_STROKE_CAP_CODE     ,"&Oslash;" ,"Oslash;" , (char)''  },     
  {IKS_ASCII_U_GRAVE_CAP           ,"&#217;"   ,"#217;"   , (char)''  },     
  {IKS_ASCII_U_GRAVE_CAP_CODE      ,"&Ugrave;" ,"Ugrave;" , (char)''  },     
  {IKS_ASCII_U_ACUTE_CAP           ,"&#218;"   ,"#218;"   , (char)''  },     
  {IKS_ASCII_U_ACUTE_CAP_CODE      ,"&Uacute;" ,"Uacute;" , (char)''  },     
  {IKS_ASCII_U_CIRCUMFLEX_CAP      ,"&#219;"   ,"#219;"   , (char)''  },
  {IKS_ASCII_U_CIRCUMFLEX_CAP_CODE ,"&Ucirc;"  ,"Ucirc;"  , (char)''  },
  {IKS_ASCII_U_DIAERESIS_CAP       ,"&#220;"   ,"#220;"   , (char)''  },
  {IKS_ASCII_U_DIAERESIS_CAP_CODE  ,"&Uuml;"   ,"Uuml;"   , (char)''  },
  {IKS_ASCII_Y_ACUTE_CAP           ,"&#221;"   ,"#221;"   , (char)''  },
  {IKS_ASCII_Y_ACUTE_CAP_CODE      ,"&Yacute;" ,"Yacute;" , (char)''  }, 
  {IKS_ASCII_THORN_CAP             ,"&#222;"   ,"#222;"   , (char)''  }, 
  {IKS_ASCII_THORN_CAP_CODE        ,"&THORN;"  ,"THORN;"  , (char)''  }, 
  {IKS_ASCII_SHARP_S               ,"&#223;"   ,"#223;"   , (char)''  }, 
  {IKS_ASCII_SHARP_S_CODE          ,"&szlig;"  ,"szlig;"  , (char)''  }, 
  {IKS_ASCII_A_GRAVE               ,"&#224;"   ,"#224;"   , (char)''  },  
  {IKS_ASCII_A_GRAVE_CODE          ,"&agrave;" ,"agrave;" , (char)''  },   
  {IKS_ASCII_A_ACUTE               ,"&#225;"   ,"#225;"   , (char)''  },  
  {IKS_ASCII_A_ACUTE_CODE          ,"&aacute;" ,"aacute;" , (char)''  },   
  {IKS_ASCII_A_CIRCUMFLEX          ,"&#226;"   ,"#226;"   , (char)''  },    
  {IKS_ASCII_A_CIRCUMFLEX_CODE     ,"&acirc;"  ,"acirc;"  , (char)''  },    
  {IKS_ASCII_A_TILDE               ,"&#227;"   ,"#227;"   , (char)''  }, 
  {IKS_ASCII_A_TILDE_CODE          ,"&atilde;" ,"atilde;" , (char)''  }, 
  {IKS_ASCII_A_DIAERESIS           ,"&#228;"   ,"#228;"   , (char)''  }, 
  {IKS_ASCII_A_DIAERESIS_CODE      ,"&auml;"   ,"auml;"   , (char)''  }, 
  {IKS_ASCII_A_RING                ,"&#229;"   ,"#229;"   , (char)''  }, 
  {IKS_ASCII_A_RING_CODE           ,"&aring;"  ,"aring;"  , (char)''  },   
  {IKS_ASCII_AE                    ,"&#230;"   ,"#230;"   , (char)''  }, 
  {IKS_ASCII_AE_CODE               ,"&aelig;"  ,"aelig;"  , (char)''  },   
  {IKS_ASCII_C_CEDILLA             ,"&#231;"   ,"#231;"   , (char)''  }, 
  {IKS_ASCII_C_CEDILLA_CODE        ,"&ccedil;" ,"ccedil;" , (char)''  }, 
  {IKS_ASCII_E_GRAVE               ,"&#232;"   ,"#232;"   , (char)''  }, 
  {IKS_ASCII_E_GRAVE_CODE          ,"&egrave;" ,"egrave;" , (char)''  }, 
  {IKS_ASCII_E_ACUTE               ,"&#233;"   ,"#233;"   , (char)''  }, 
  {IKS_ASCII_E_ACUTE_CODE          ,"&eacute;" ,"eacute;" , (char)''  }, 
  {IKS_ASCII_E_CIRCUMFLEX          ,"&#234;"   ,"#234;"   , (char)''  }, 
  {IKS_ASCII_E_CIRCUMFLEX_CODE     ,"&ecirc;"  ,"ecirc;"  , (char)''  },  
  {IKS_ASCII_E_DIAERESIS           ,"&#235;"   ,"#235;"   , (char)''  },  
  {IKS_ASCII_E_DIAERESIS_CODE      ,"&euml;"   ,"euml;"   , (char)''  },   
  {IKS_ASCII_I_GRAVE               ,"&#236;"   ,"#236;"   , (char)''  },  
  {IKS_ASCII_I_GRAVE_CODE          ,"&igrave;" ,"igrave;" , (char)''  },   
  {IKS_ASCII_I_ACUTE               ,"&#237;"   ,"#237;"   , (char)''  },  
  {IKS_ASCII_I_ACUTE_CODE          ,"&iacute;" ,"iacute;" , (char)''  },   
  {IKS_ASCII_I_CIRCUMFLEX          ,"&#238;"   ,"#238;"   , (char)''  },  
  {IKS_ASCII_I_CIRCUMFLEX_CODE     ,"&icirc;"  ,"icirc;"  , (char)''  },   
  {IKS_ASCII_I_DIAERESIS           ,"&#239;"   ,"#239;"   , (char)''  },  
  {IKS_ASCII_I_DIAERESIS_CODE      ,"&iuml;"   ,"iuml;"   , (char)''  },  
  {IKS_ASCII_ETH                   ,"&#240;"   ,"#240;"   , (char)''  },  
  {IKS_ASCII_ETH_CODE              ,"&eth;"    ,"eth;"    , (char)''  },  
  {IKS_ASCII_N_TILDE               ,"&#241;"   ,"#241;"   , (char)''  },  
  {IKS_ASCII_N_TILDE_CODE          ,"&ntilde;" ,"ntilde;" , (char)''  },  
  {IKS_ASCII_O_GRAVE               ,"&#242;"   ,"#242;"   , (char)''  },  
  {IKS_ASCII_O_GRAVE_CODE          ,"&ograve;" ,"ograve;" , (char)''  },  
  {IKS_ASCII_O_ACUTE               ,"&#243;"   ,"#243;"   , (char)''  },  
  {IKS_ASCII_O_ACUTE_CODE          ,"&oacute;" ,"oacute;" , (char)''  },  
  {IKS_ASCII_O_CIRCUMFLEX          ,"&#244;"   ,"#244;"   , (char)''  },  
  {IKS_ASCII_O_CIRCUMFLEX_CODE     ,"&ocirc;"  ,"ocirc;"  , (char)''  },  
  {IKS_ASCII_O_TILDE               ,"&#245;"   ,"#245;"   , (char)''  },  
  {IKS_ASCII_O_TILDE_CODE          ,"&otilde;" ,"otilde;" , (char)''  },   
  {IKS_ASCII_O_DIAERESIS           ,"&#246;"   ,"#246;"   , (char)''  },  
  {IKS_ASCII_O_DIAERESIS_CODE      ,"&ouml;"   ,"ouml;"   , (char)''  },  
  {IKS_ASCII_DIVISION              ,"&#247;"   ,"#247;"   , (char)''  },  
  {IKS_ASCII_O_STROKE              ,"&#248;"   ,"#248;"   , (char)''  },  
  {IKS_ASCII_O_STROKE_CODE         ,"&oslash;" ,"oslash;" , (char)''  },  
  {IKS_ASCII_U_GRAVE               ,"&#249;"   ,"#249;"   , (char)''  },  
  {IKS_ASCII_U_GRAVE_CODE          ,"&ugrave;" ,"ugrave;" , (char)''  },  
  {IKS_ASCII_U_ACUTE               ,"&#250;"   ,"#250;"   , (char)''  },  
  {IKS_ASCII_U_ACUTE_CODE          ,"&uacute;" ,"uacute;" , (char)''  },  
  {IKS_ASCII_U_CIRCUMFLEX          ,"&#251;"   ,"#251;"   , (char)''  },  
  {IKS_ASCII_U_CIRCUMFLEX_CODE     ,"&ucirc;"  ,"ucirc;"  , (char)''  },  
  {IKS_ASCII_U_DIAERESIS           ,"&#252;"   ,"#252;"   , (char)''  }, 
  {IKS_ASCII_U_DIAERESIS_CODE      ,"&uuml;"   ,"uuml;"   , (char)''  }, 
  {IKS_ASCII_Y_ACUTE               ,"&#253;"   ,"#253;"   , (char)''  },  
  {IKS_ASCII_Y_ACUTE_CODE          ,"&yacute;" ,"yacute;" , (char)''  },  
  {IKS_ASCII_THORN                 ,"&#254;"   ,"#254;"   , (char)''  },  
  {IKS_ASCII_THORN_CODE            ,"&thorn;"  ,"thorn;"  , (char)''  },  
  {IKS_ASCII_Y_DIAERESIS           ,"&#255;"   ,"#255;"   , (char)''  },  
  {IKS_ASCII_Y_DIAERESIS_CODE      ,"&yuml;"   ,"yuml;"   , (char)''  },  
  {IKS_ASCII_UNKNOWN     ,"" ,""   , (char)' '  }
};
#endif

/* if you add a variable here, dont forget changing iks_parser_reset */
struct iksparser_struct {
	ikstack *s;
	void *user_data;
	iksTagHook *tagHook;
	iksCDataHook *cdataHook;
	iksDeleteHook *deleteHook;
	/* parser context */
	char *stack;
	size_t stack_pos;
	size_t stack_max;

	enum cons_e context;
	enum cons_e oldcontext;

	char *tag_name;
	enum ikstagtype tagtype;

	unsigned int attmax;
	unsigned int attcur;
	int attflag;
	char **atts;
	int valflag;

	unsigned int entpos;
	char entity[8];

	unsigned long nr_bytes;
	unsigned long nr_lines;

	int uni_max;
	int uni_len;
};

#ifdef SHC_VERSION
IKS_ASCII_NO iks_sax_eGetAsciiCode
( 
  char *pcName
)
/* LONG-NAME    : returns ASCII Code no
 * AUTHOR       : NIJMAN
 * LAST UPDATE  : 04.09.2008
 * PRECONDITION : ###
 * POSTCONDITION: IKS_ASCII_UNKNOWN when Ascii code not found
 * ERROR EXITS  : ###
 * SIDE EFFECTS : ###
 **************************************************************************/
{
  IKS_ASCII_NO eNo;
  unsigned short uiI;
  unsigned short uiAsciiCodeLength;

  uiI = 0;

  do
  {
    eNo = astAsciiCodeDefinitions[uiI].eNo;
    uiAsciiCodeLength = strlen(astAsciiCodeDefinitions[uiI].pcAsciiCode);
    if (strncmp(pcName, astAsciiCodeDefinitions[uiI].pcAsciiCode, uiAsciiCodeLength) == 0)
    {
      break;
    }
    uiI++;
  } while (eNo != IKS_ASCII_UNKNOWN);

  return eNo;
}

unsigned char iks_sax_bAsciiCodingText
(
  char *pcText
)
/* LONG-NAME    : Check string for ASCII coding
 * AUTHOR       : NIJMAN
 * LAST UPDATE  : 04.09.2008
 * PRECONDITION : ###
 * POSTCONDITION: Found ASCII coding replaced by related character
 * ERROR EXITS  : ###
 * SIDE EFFECTS : ###
 **************************************************************************/
{
  IKS_ASCII_NO eNo;
  unsigned char bFoundAsciiCode=0x00;

  eNo = iks_sax_eGetAsciiCode(&pcText[0]);
  if (eNo != IKS_ASCII_UNKNOWN)
  {
    if ((eNo != IKS_ASCII_EURO) && 
        (eNo != IKS_ASCII_EURO_CODE) &&
        (eNo != IKS_ASCII_EURO_CAP_CODE) &&
        (eNo != IKS_ASCII_EURO_NUM_CODE))
    {
      pcText[0]=astAsciiCodeDefinitions[eNo].cChar;
      pcText[1]='\0';
    }
    else
    {
      pcText[0]=(char) 0x8F;  // "E282AC
      pcText[1]=(char) 0x01;
      pcText[2]=(char) 0xF0;
      pcText[3]='\0';
    }
    bFoundAsciiCode=0x01;
  }
  return bFoundAsciiCode;
}
#endif

iksparser *
iks_sax_new (void *user_data, iksTagHook *tagHook, iksCDataHook *cdataHook)
{
	iksparser *prs;

#ifdef SHC_VERSION	
  prs = (iksparser *) iks_malloc (sizeof (iksparser));
#else  
  prs = iks_malloc (sizeof (iksparser));
#endif
	if (NULL == prs) return NULL;
	memset (prs, 0, sizeof (iksparser));
	prs->user_data = user_data;
	prs->tagHook = tagHook;
	prs->cdataHook = cdataHook;
	return prs;
}

iksparser *
iks_sax_extend (ikstack *s, void *user_data, iksTagHook *tagHook, iksCDataHook *cdataHook, iksDeleteHook *deleteHook)
{
	iksparser *prs;

#ifdef SHC_VERSION	
  prs = (iksparser *) iks_stack_alloc (s, sizeof (iksparser));
#else  
  prs = iks_stack_alloc (s, sizeof (iksparser));
#endif
	if (NULL == prs) return NULL;
	memset (prs, 0, sizeof (iksparser));
	prs->s = s;
	prs->user_data = user_data;
	prs->tagHook = tagHook;
	prs->cdataHook = cdataHook;
	prs->deleteHook = deleteHook;
	return prs;
}

#ifndef SHC_VERSION
ikstack *
iks_parser_stack (iksparser *prs)
{
	return prs->s;
}
#endif

#ifndef SHC_VERSION
void *
iks_user_data (iksparser *prs)
{
	return prs->user_data;
}
#endif

unsigned long
iks_nr_bytes (iksparser *prs)
{
	return prs->nr_bytes;
}

#ifndef SHC_VERSION
unsigned long
iks_nr_lines (iksparser *prs)
{
	return prs->nr_lines;
}
#endif

#define IS_WHITESPACE(x) ' ' == (x) || '\t' == (x) || '\r' == (x) || '\n' == (x)
#define NOT_WHITESPACE(x) ' ' != (x) && '\t' != (x) && '\r' != (x) && '\n' != (x)

static int
stack_init (iksparser *prs)
{
#ifdef SHC_VERSION	
  prs->stack = (char *) iks_malloc (128);
#else  
  prs->stack = iks_malloc (128);
#endif
	if (!prs->stack) return 0;
	prs->stack_max = 128;
	prs->stack_pos = 0;
	return 1;
}

static int
stack_expand (iksparser *prs, int len)
{
	size_t need;
#ifdef SHC_VERSION	
  long diff;
#else  
  off_t diff;
#endif
	char *tmp;
	need = len - (prs->stack_max - prs->stack_pos);
	if (need < prs->stack_max) {
		need = prs->stack_max * 2;
	} else {
#ifdef SHC_VERSION
    need = prs->stack_max + (size_t) (need * 1.2);
    }
	  tmp = (char *) iks_malloc (need);
#else
    need = prs->stack_max + (need * 1.2);
    }
	  tmp = iks_malloc (need);
#endif
	if (!tmp) return 0;
	diff = tmp - prs->stack;
	memcpy (tmp, prs->stack, prs->stack_max);
	iks_free (prs->stack);
	prs->stack = tmp;
	prs->stack_max = need;
	prs->tag_name += diff;
	if (prs->attflag != 0) {
#ifdef SHC_VERSION		
    unsigned int i = 0;
#else    
    int i = 0;
#endif
		while (i < (prs->attmax * 2)) {
			if (prs->atts[i]) prs->atts[i] += diff;
			i++;
		}
	}
	return 1;
}

#define STACK_INIT \
	if (NULL == prs->stack && 0 == stack_init (prs)) return IKS_NOMEM

#define STACK_PUSH_START (prs->stack + prs->stack_pos)

#define STACK_PUSH(buf,len) \
{ \
	char *sbuf = (buf); \
	size_t slen = (len); \
	if (prs->stack_max - prs->stack_pos <= slen) { \
		if (0 == stack_expand (prs, slen)) return IKS_NOMEM; \
	} \
	memcpy (prs->stack + prs->stack_pos, sbuf, slen); \
	prs->stack_pos += slen; \
}

#define STACK_PUSH_END \
{ \
	if (prs->stack_pos >= prs->stack_max) { \
		if (0 == stack_expand (prs, 1)) return IKS_NOMEM; \
	} \
	prs->stack[prs->stack_pos] = '\0'; \
	prs->stack_pos++; \
}

static enum ikserror
sax_core (iksparser *prs, char *buf, int len)
{
	enum ikserror err;
	int pos = 0, old = 0, re, stack_old = -1;
	unsigned char c;

	while (pos < len) {
		re = 0;
		c = buf[pos];
		if (0 == c || 0xFE == c || 0xFF == c) return IKS_BADXML;
		if (prs->uni_max) {
			if ((c & 0xC0) != 0x80) return IKS_BADXML;
			prs->uni_len++;
			if (prs->uni_len == prs->uni_max) prs->uni_max = 0;
			goto cont;
		} else {
			if (c & 0x80) {
				unsigned char mask;
				if ((c & 0x60) == 0x40) {
					prs->uni_max = 2;
					mask = 0x1F;
				} else if ((c & 0x70) == 0x60) {
					prs->uni_max = 3;
					mask = 0x0F;
				} else if ((c & 0x78) == 0x70) {
					prs->uni_max = 4;
					mask = 0x07;
				} else if ((c & 0x7C) == 0x78) {
					prs->uni_max = 5;
					mask = 0x03;
				} else if ((c & 0x7E) == 0x7C) {
					prs->uni_max = 6;
					mask = 0x01;
				} else {
					return IKS_BADXML;
				}
#ifndef SHC_VERSION						
				if ((c & mask) == 0) 
        {
          return IKS_BADXML;
        }
#endif
				prs->uni_len = 1;
				if (stack_old == -1
					&& (prs->context == C_TAG
						|| prs->context == C_ATTRIBUTE_1
						|| prs->context == C_VALUE_APOS
						|| prs->context == C_VALUE_QUOT)) stack_old = pos;
				goto cont;
			}
		}

		switch (prs->context) {
			case C_CDATA:
				if ('&' == c) {
					if (old < pos && prs->cdataHook) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, &buf[old], pos - old);
#else            
            err = prs->cdataHook (prs->user_data, &buf[old], pos - old);
#endif
						if (IKS_OK != err) 
            {
              return err;
            }
					}
					prs->context = C_ENTITY;
					prs->entpos = 0;
#ifdef SHC_VERSION						
					prs->entity[prs->entpos++] = buf[pos];
#endif
					break;
				}
				if ('<' == c) {
					if (old < pos && prs->cdataHook) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, &buf[old], pos - old);
#else            
            err = prs->cdataHook (prs->user_data, &buf[old], pos - old);
#endif						
            if (IKS_OK != err) 
            {
              return err;
            }
					}
					STACK_INIT;
					prs->tag_name = STACK_PUSH_START;
					if (!prs->tag_name) 
          {
            return IKS_NOMEM;
          }
					prs->context = C_TAG_START;
				}
				break;

			case C_TAG_START:
				prs->context = C_TAG;
				if ('/' == c) {
					prs->tagtype = IKS_CLOSE;
					break;
				}
				if ('?' == c) {
					prs->context = C_PI;
					break;
				}
				if ('!' == c) {
					prs->context = C_MARKUP;
					break;
				}
				prs->tagtype = IKS_OPEN;
				stack_old = pos;
				break;

			case C_TAG:
				if (IS_WHITESPACE(c)) {
					if (IKS_CLOSE == prs->tagtype)
						prs->oldcontext = C_TAG_END;
					else
						prs->oldcontext = C_ATTRIBUTE;
					prs->context = C_WHITESPACE;
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					break;
				}
				if ('/' == c) {
					if (IKS_CLOSE == prs->tagtype) 
          {
            return IKS_BADXML;
          }
					prs->tagtype = IKS_SINGLE;
					prs->context = C_TAG_END;
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					break;
				}
				if ('>' == c) {
					prs->context = C_TAG_END;
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					re = 1;
					break;
				}
				if (stack_old == -1) stack_old = pos;
				break;

			case C_TAG_END:
				if (c != '>') 
        {
          return IKS_BADXML;
        }
				if (prs->tagHook) {
					char **tmp;
					if (prs->attcur == 0) tmp = NULL; else tmp = prs->atts;
#ifdef SHC_VERSION					
          err = (enum ikserror) prs->tagHook (prs->user_data, prs->tag_name, tmp, prs->tagtype);
#else     
          err = prs->tagHook (prs->user_data, prs->tag_name, tmp, prs->tagtype);
#endif
					if (IKS_OK != err) 
          {
            return err;
          }
				}
				prs->stack_pos = 0;
				stack_old = -1;
				prs->attcur = 0;
				prs->attflag = 0;
				prs->context = C_CDATA;
				old = pos + 1;
				break;

			case C_ATTRIBUTE:
				if ('/' == c) {
					prs->tagtype = IKS_SINGLE;
					prs->context = C_TAG_END;
					break;
				}
				if ('>' == c) {
					prs->context = C_TAG_END;
					re = 1;
					break;
				}
				if (!prs->atts) {
					prs->attmax = 12;
#ifdef SHC_VERSION					
          prs->atts = (char **) iks_malloc (sizeof(char *) * 2 * 12);
#else          
          prs->atts = iks_malloc (sizeof(char *) * 2 * 12);
#endif
					if (!prs->atts) 
          {
            return IKS_NOMEM;
          }
					memset (prs->atts, 0, sizeof(char *) * 2 * 12);
					prs->attcur = 0;
				} else {
					if (prs->attcur >= (prs->attmax * 2)) {
						void *tmp;
						prs->attmax += 12;
						tmp = iks_malloc (sizeof(char *) * 2 * prs->attmax);
						if (!tmp) 
            {
              return IKS_NOMEM;
            }
						memset (tmp, 0, sizeof(char *) * 2 * prs->attmax);
						memcpy (tmp, prs->atts, sizeof(char *) * prs->attcur);
						iks_free (prs->atts);
#ifdef SHC_VERSION						
            prs->atts = (char **) tmp;
#else            
            prs->atts = tmp;
#endif
					}
				}
				prs->attflag = 1;
				prs->atts[prs->attcur] = STACK_PUSH_START;
				stack_old = pos;
				prs->context = C_ATTRIBUTE_1;
				break;

			case C_ATTRIBUTE_1:
				if ('=' == c) {
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					prs->context = C_VALUE;
					break;
				}
				if (stack_old == -1) stack_old = pos;
				break;

			case C_ATTRIBUTE_2:
				if ('/' == c) {
					prs->tagtype = IKS_SINGLE;
					prs->atts[prs->attcur] = NULL;
					prs->context = C_TAG_END;
					break;
				}
				if ('>' == c) {
					prs->atts[prs->attcur] = NULL;
					prs->context = C_TAG_END;
					re = 1;
					break;
				}
				prs->context = C_ATTRIBUTE;
				re = 1;
				break;

			case C_VALUE:
				prs->atts[prs->attcur + 1] = STACK_PUSH_START;
				if ('\'' == c) {
					prs->context = C_VALUE_APOS;
					break;
				}
				if ('"' == c) {
					prs->context = C_VALUE_QUOT;
					break;
				}
				return IKS_BADXML;

			case C_VALUE_APOS:
				if ('\'' == c) {
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					prs->oldcontext = C_ATTRIBUTE_2;
					prs->context = C_WHITESPACE;
					prs->attcur += 2;
				}
				if (stack_old == -1) stack_old = pos;
				break;

			case C_VALUE_QUOT:
				if ('"' == c) {
					if (stack_old != -1) STACK_PUSH (buf + stack_old, pos - stack_old);
					stack_old = -1;
					STACK_PUSH_END;
					prs->oldcontext = C_ATTRIBUTE_2;
					prs->context = C_WHITESPACE;
					prs->attcur += 2;
				}
				if (stack_old == -1) stack_old = pos;
				break;

			case C_WHITESPACE:
				if (NOT_WHITESPACE(c)) {
					prs->context = prs->oldcontext;
					re = 1;
				}
				break;

			case C_ENTITY:
#ifdef SHC_VERSION
        switch (c)
        {
				char hede[7]; // "Must be able to contain a complete ASCII coding string"
        case ';':

          prs->entity[prs->entpos++] = buf[pos];
          prs->entity[prs->entpos] = '\0';
          strcpy(hede,prs->entity);
          iks_sax_bAsciiCodingText(hede);
#else
         if (';' == c) {

					  char hede[2];

            prs->entity[prs->entpos] = '\0';
					  if (strcmp(prs->entity, "amp") == 0)
						  t = '&';
					  else if (strcmp(prs->entity, "quot") == 0)
						  t = '"';
					  else if (strcmp(prs->entity, "apos") == 0)
						  t = '\'';
					  else if (strcmp(prs->entity, "lt") == 0)
						  t = '<';
					  else if (strcmp(prs->entity, "gt") == 0)
						  t = '>';
					  hede[0] = '?';
#endif
					  old = pos + 1;
					  if (prs->cdataHook) {
#ifdef SHC_VERSION						
              err = (enum ikserror) prs->cdataHook (prs->user_data, &hede[0], strlen(hede));
#else            
              err = prs->cdataHook (prs->user_data, &hede[0], 1);
#endif
						  if (IKS_OK != err) 
              {
                return err;
              }
					  }
					prs->context = C_CDATA;
          break;
        case '<':
 					old = pos - (prs->entpos++);
					if (old < pos && prs->cdataHook) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, &buf[old], pos - old);
#else            
            err = prs->cdataHook (prs->user_data, &buf[old], pos - old);
#endif						
            if (IKS_OK != err) 
            {
              return err;
            }
					}
					STACK_INIT;
					prs->tag_name = STACK_PUSH_START;
					if (!prs->tag_name) 
          {
            return IKS_NOMEM;
          }
					prs->context = C_TAG_START;
          break;
        case ' ':
 					old = pos - (prs->entpos++);
					prs->context = C_CDATA;
          break;
        default:
          prs->entity[prs->entpos++] = buf[pos];
					if (prs->entpos > 7) 
          {
   					old = pos - (prs->entpos++);
  					prs->context = C_CDATA;
          }
          break;
				}
				break;

			case C_COMMENT:
				if ('-' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_COMMENT_1;
				break;

			case C_COMMENT_1:
				if ('-' == c) prs->context = C_COMMENT_2;
				break;

			case C_COMMENT_2:
				if ('-' == c)
					prs->context = C_COMMENT_3;
				else
					prs->context = C_COMMENT_1;
				break;

			case C_COMMENT_3:
				if ('>' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_CDATA;
				old = pos + 1;
				break;

			case C_MARKUP:
				if ('[' == c) {
					prs->context = C_SECT;
					break;
				}
				if ('-' == c) {
					prs->context = C_COMMENT;
					break;
				}
				prs->context = C_MARKUP_1;

			case C_MARKUP_1:
				if ('>' == c) {
					old = pos + 1;
					prs->context = C_CDATA;
				}
				break;

			case C_SECT:
				if ('C' == c) {
					prs->context = C_SECT_CDATA;
					break;
				}
				return IKS_BADXML;

			case C_SECT_CDATA:
				if ('D' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_SECT_CDATA_1;
				break;

			case C_SECT_CDATA_1:
				if ('A' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_SECT_CDATA_2;
				break;

			case C_SECT_CDATA_2:
				if ('T' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_SECT_CDATA_3;
				break;

			case C_SECT_CDATA_3:
				if ('A' != c) 
        {
          return IKS_BADXML;
        }
				prs->context = C_SECT_CDATA_4;
				break;

			case C_SECT_CDATA_4:
				if ('[' != c) 
        {
          return IKS_BADXML;
        }
				old = pos + 1;
				prs->context = C_SECT_CDATA_C;
				break;

			case C_SECT_CDATA_C:
				if (']' == c) {
					prs->context = C_SECT_CDATA_E;
					if (prs->cdataHook && old < pos) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, &buf[old], pos - old);
#else            
            err = prs->cdataHook (prs->user_data, &buf[old], pos - old);
#endif
						if (IKS_OK != err) 
            {
              return err;
            }
					}
				}
				break;

			case C_SECT_CDATA_E:
				if (']' == c) {
					prs->context = C_SECT_CDATA_E2;
				} else {
					if (prs->cdataHook) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, "]", 1);
#else            
            err = prs->cdataHook (prs->user_data, "]", 1);
#endif
						if (IKS_OK != err) 
            {
              return err;
            }
					}
					old = pos;
					prs->context = C_SECT_CDATA_C;
				}
				break;

			case C_SECT_CDATA_E2:
				if ('>' == c) {
					old = pos + 1;
					prs->context = C_CDATA;
				} else {
					if (prs->cdataHook) {
#ifdef SHC_VERSION						
            err = (enum ikserror) prs->cdataHook (prs->user_data, "]]", 2);
#else            
            err = prs->cdataHook (prs->user_data, "]]", 2);
#endif
						if (IKS_OK != err) 
            {
              return err;
            }
					}
					old = pos;
					prs->context = C_SECT_CDATA_C;
				}
				break;

			case C_PI:
				old = pos + 1;
				if ('>' == c) prs->context = C_CDATA;
				break;
		}
cont:
		if (0 == re) {
			pos++;
			prs->nr_bytes++;
			if ('\n' == c) prs->nr_lines++;
		}
	}

	if (stack_old != -1)
		STACK_PUSH (buf + stack_old, pos - stack_old);

	err = IKS_OK;
	if (prs->cdataHook && (prs->context == C_CDATA || prs->context == C_SECT_CDATA_C) && old < pos)
#ifdef SHC_VERSION		
    err = (enum ikserror) prs->cdataHook (prs->user_data, &buf[old], pos - old);
#else    
    err = prs->cdataHook (prs->user_data, &buf[old], pos - old);
#endif
	return err;
}

int
iks_parse (iksparser *prs, const char *data, size_t len, int finish)
{
	if (!data) return IKS_OK;
	if (len == 0) len = strlen (data);
	return sax_core (prs, (char *) data, len);
}

void
iks_parser_reset (iksparser *prs)
{
	if (prs->deleteHook) prs->deleteHook (prs->user_data);
	prs->stack_pos = 0;
#ifdef SHC_VERSION	
  prs->context = (enum cons_e) 0;
	prs->oldcontext = (enum cons_e) 0;
	prs->tagtype = (enum ikstagtype) 0;
#else
  prs->context = 0;
	prs->oldcontext = 0;
	prs->tagtype = 0;
#endif
	prs->attcur = 0;
	prs->attflag = 0;
	prs->valflag = 0;
	prs->entpos = 0;
	prs->nr_bytes = 0;
	prs->nr_lines = 0;
	prs->uni_max = 0;
	prs->uni_len = 0;
}

void
iks_parser_delete (iksparser *prs)
{
	if (prs->deleteHook) prs->deleteHook (prs->user_data);
	if (prs->stack) iks_free (prs->stack);
	if (prs->atts) iks_free (prs->atts);
	if (prs->s) iks_stack_delete (prs->s); else iks_free (prs);
}
